/*
 * Decompiled with CFR 0.152.
 */
package org.ros.time;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Queue;
import java.util.concurrent.Callable;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ros.exception.RosRuntimeException;
import org.ros.time.TimeProvider;

public class RemoteUptimeClock {
    private static final boolean DEBUG = false;
    private static final Log log = LogFactory.getLog(RemoteUptimeClock.class);
    private final LocalUptimeProvider localUptimeProvider;
    private final Callable<Double> callable;
    private final LatencyOutlierFilter latencyOutlierFilter;
    private final double driftSensitivity;
    private final double errorReductionCoefficientSensitivity;
    private double localUptime;
    private double measuredRemoteUptime;
    private double predictedRemoteUptime;
    private double drift;
    private double errorReductionCoefficient;

    public static RemoteUptimeClock newDefault(final TimeProvider timeProvider, Callable<Double> callable, double driftSensitivity, double errorReductionCoefficientSensitivity, int latencyOutlierFilterSampleSize, double latencyOutlierFilterThreshold) {
        return new RemoteUptimeClock(new LocalUptimeProvider(){

            @Override
            public double getSeconds() {
                return timeProvider.getCurrentTime().toSeconds();
            }
        }, callable, driftSensitivity, errorReductionCoefficientSensitivity, latencyOutlierFilterSampleSize, latencyOutlierFilterThreshold);
    }

    @VisibleForTesting
    RemoteUptimeClock(LocalUptimeProvider localUptimeProvider, Callable<Double> callable, double driftSensitivity, double errorReductionCoefficientSensitivity, int latencyOutlierFilterSampleSize, double latencyOutlierFilterThreshold) {
        Preconditions.checkArgument((driftSensitivity >= 0.0 && driftSensitivity <= 1.0 ? 1 : 0) != 0);
        Preconditions.checkArgument((errorReductionCoefficientSensitivity >= 0.0 && errorReductionCoefficientSensitivity <= 1.0 ? 1 : 0) != 0);
        this.localUptimeProvider = localUptimeProvider;
        this.callable = callable;
        this.driftSensitivity = driftSensitivity;
        this.errorReductionCoefficientSensitivity = errorReductionCoefficientSensitivity;
        this.latencyOutlierFilter = new LatencyOutlierFilter(latencyOutlierFilterSampleSize, latencyOutlierFilterThreshold);
        this.errorReductionCoefficient = 0.0;
    }

    public void calibrate(int sampleSize, double samplingDelayMillis) {
        log.info((Object)"Starting calibration...");
        double remoteUptimeSum = 0.0;
        double localUptimeSum = 0.0;
        double driftSum = 0.0;
        for (int i = 0; i < sampleSize; ++i) {
            UptimeCalculationResult result = this.calculateNewUptime(this.callable);
            this.latencyOutlierFilter.add(result.latency);
            if (i > 0) {
                double localUptimeDelta = result.newLocalUptime - this.localUptime;
                double remoteUptimeDelta = result.newRemoteUptime - this.measuredRemoteUptime;
                driftSum += this.calculateDrift(localUptimeDelta, remoteUptimeDelta);
            }
            this.measuredRemoteUptime = result.newRemoteUptime;
            this.localUptime = result.newLocalUptime;
            remoteUptimeSum += this.measuredRemoteUptime;
            localUptimeSum += this.localUptime;
            try {
                Thread.sleep((long)samplingDelayMillis);
                continue;
            }
            catch (InterruptedException e) {
                throw new RosRuntimeException(e);
            }
        }
        this.drift = driftSum / (double)(sampleSize - 1);
        double offset = (this.drift * remoteUptimeSum - localUptimeSum) / (double)sampleSize;
        this.predictedRemoteUptime = (this.localUptime + offset) / this.drift;
        log.info((Object)String.format("Calibration complete. Drift: %.4g, Offset: %.4f s", this.drift, offset));
    }

    private double calculateDrift(double localUptimeDelta, double remoteUptimeDelta) {
        Preconditions.checkState((remoteUptimeDelta > 1.0E-9 ? 1 : 0) != 0);
        return localUptimeDelta / remoteUptimeDelta;
    }

    public void update() {
        UptimeCalculationResult result = this.calculateNewUptime(this.callable);
        double newLocalUptime = result.newLocalUptime;
        double newRemoteUptime = result.newRemoteUptime;
        double latency = result.latency;
        if (this.latencyOutlierFilter.add(latency)) {
            log.warn((Object)String.format("Measurement latency marked as outlier. Latency: %.4f s, Median: %.4f s", latency, this.latencyOutlierFilter.getMedian()));
            return;
        }
        double localUptimeDelta = newLocalUptime - this.localUptime;
        double remoteUptimeDelta = newRemoteUptime - this.measuredRemoteUptime;
        Preconditions.checkState((localUptimeDelta > 1.0E-9 ? 1 : 0) != 0);
        Preconditions.checkState((remoteUptimeDelta > 1.0E-9 ? 1 : 0) != 0);
        double newDrift = this.driftSensitivity * (localUptimeDelta / remoteUptimeDelta) + (1.0 - this.driftSensitivity) * this.drift;
        double newPredictedRemoteUptime = this.predictedRemoteUptime + localUptimeDelta / (this.drift + this.errorReductionCoefficient);
        double nextPredictedRemoteUptime = newRemoteUptime + remoteUptimeDelta;
        double newCombinedDriftAndError = localUptimeDelta / (nextPredictedRemoteUptime - newPredictedRemoteUptime);
        double newErrorReductionCoefficient = this.errorReductionCoefficientSensitivity * (newCombinedDriftAndError - newDrift);
        double deltaRatio = remoteUptimeDelta / localUptimeDelta;
        double error = newLocalUptime - this.toLocalUptime(newRemoteUptime);
        log.info((Object)String.format("Latency: %.4f s, Delta ratio: %.4f, Drift: %.4g, Error reduction coefficient: %.4g, Error: %.4f s", latency, deltaRatio, newDrift, newErrorReductionCoefficient, error));
        this.measuredRemoteUptime = newRemoteUptime;
        this.predictedRemoteUptime = newPredictedRemoteUptime;
        this.localUptime = newLocalUptime;
        this.drift = newDrift;
        this.errorReductionCoefficient = newErrorReductionCoefficient;
    }

    private UptimeCalculationResult calculateNewUptime(Callable<Double> callable) {
        double newRemoteUptime;
        double newLocalUptime = this.localUptimeProvider.getSeconds();
        try {
            newRemoteUptime = callable.call();
        }
        catch (Exception e) {
            log.error((Object)e);
            throw new RosRuntimeException(e);
        }
        double latency = this.localUptimeProvider.getSeconds() - newLocalUptime;
        double latencyOffset = latency / 2.0;
        return new UptimeCalculationResult(newLocalUptime += latencyOffset, newRemoteUptime, latency);
    }

    public double toLocalUptime(double remoteUptime) {
        double localOffset = (this.drift + this.errorReductionCoefficient) * (remoteUptime - this.predictedRemoteUptime);
        return this.localUptime + localOffset;
    }

    @VisibleForTesting
    double getDrift() {
        return this.drift;
    }

    @VisibleForTesting
    double getErrorReductionCoefficient() {
        return this.errorReductionCoefficient;
    }

    @VisibleForTesting
    static interface LocalUptimeProvider {
        public double getSeconds();
    }

    private final class LatencyOutlierFilter {
        private final int sampleSize;
        private final double threshold;
        private final Queue<Double> latencies;

        public LatencyOutlierFilter(int sampleSize, double threshold) {
            Preconditions.checkArgument((sampleSize > 0 ? 1 : 0) != 0);
            Preconditions.checkArgument((threshold > 1.0 ? 1 : 0) != 0);
            this.threshold = threshold;
            this.sampleSize = sampleSize;
            this.latencies = Lists.newLinkedList();
        }

        public boolean add(double latency) {
            this.latencies.add(latency);
            if (this.latencies.size() <= this.sampleSize) {
                return false;
            }
            this.latencies.remove();
            double medianLatency = this.getMedian();
            return !(latency < medianLatency * this.threshold);
        }

        public double getMedian() {
            ArrayList ordered = Lists.newArrayList(this.latencies);
            Collections.sort(ordered);
            return (Double)ordered.get(this.latencies.size() / 2);
        }
    }

    private final class UptimeCalculationResult {
        final double newLocalUptime;
        final double newRemoteUptime;
        final double latency;

        public UptimeCalculationResult(double newLocalUptime, double newRemoteUptime, double latency) {
            this.newLocalUptime = newLocalUptime;
            this.newRemoteUptime = newRemoteUptime;
            this.latency = latency;
        }
    }
}

